<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers\Admin;

use Skyboard\Application\Services\Admin\NotificationRichAdminService;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class NotificationRichAdminController
{
    use AdminControllerTrait;

    public function __construct(private readonly NotificationRichAdminService $notifications)
    {
    }

    public function list(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) return $response;
        $cat = isset($request->query['categoryId']) ? (int) $request->query['categoryId'] : null;
        return Response::ok(['notifications' => $this->notifications->list($cat)]);
    }

    public function create(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) return $response;
        try {
            $notification = $this->notifications->create($request->body ?? []);
        } catch (\Throwable $e) {
            return Response::error('CREATE_FAILED', $e->getMessage(), [], 422);
        }
        return Response::ok(['notification' => $notification], 201);
    }

    public function update(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) return $response;
        $body = $request->body ?? [];
        $id = isset($body['id']) ? (int) $body['id'] : 0;
        if ($id <= 0) return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        try {
            $notification = $this->notifications->update($id, $body);
        } catch (\RuntimeException $e) {
            return Response::error('NOT_FOUND', $e->getMessage(), [], 404);
        } catch (\Throwable $e) {
            return Response::error('UPDATE_FAILED', $e->getMessage(), [], 422);
        }
        return Response::ok(['notification' => $notification]);
    }

    public function delete(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) return $response;
        $id = isset($request->body['id']) ? (int) $request->body['id'] : 0;
        if ($id <= 0) return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        $this->notifications->delete($id);
        return Response::ok(['deleted' => true]);
    }

    public function get(Request $request): Response
    {
        if ($response = $this->requireAdmin($request)) return $response;
        $raw = $request->getAttribute('id');
        $id = is_string($raw) ? (int) $raw : (int) $raw;
        if ($id <= 0) return Response::error('INVALID_PAYLOAD', 'Identifiant invalide.', [], 422);
        try {
            $notification = $this->notifications->get($id);
        } catch (\RuntimeException $e) {
            return Response::error('NOT_FOUND', 'Notification introuvable.', [], 404);
        }
        return Response::ok(['notification' => $notification]);
    }
}
