<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers;

use Skyboard\Application\Services\BoardCatalog;
use Skyboard\Application\Services\BoardRepository;
use Skyboard\Application\Services\BoardRulesService;
use Skyboard\Application\Services\TagSuggestionService;
use Skyboard\Domain\Boards\BoardConflictException;
use Skyboard\Domain\Boards\BoardState;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;

final class BoardController
{
    public function __construct(
        private readonly BoardCatalog $catalog,
        private readonly BoardRepository $boards,
        private readonly TagSuggestionService $tagSuggestions,
        private readonly BoardRulesService $boardRules,
        private readonly string $rulesVersion
    ) {
    }

    public function list(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }
        $boards = $this->catalog->listForUser((int) $user['id']);
        return Response::ok(['boards' => $boards]);
    }

    // Removed: create/rename/delete endpoints migrated to /api/commands (Non-Board Bus)

    public function state(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }
        $boardId = $this->resolveBoardId($request);
        if ($boardId === null) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }
        $snapshot = $this->boards->getSnapshot((string) $boardId, (int) $user['id']);
        if ($snapshot === null) {
            return Response::error('BOARD_NOT_FOUND', 'Board introuvable.', [], 404);
        }
        return Response::ok([
            'state' => $snapshot->state()->toArray(),
            'meta' => [
                'updatedAt' => $snapshot->updatedAt(),
                'revision' => $snapshot->revision(),
                'history' => $snapshot->history(),
            ],
        ]);
    }

    public function allowedTags(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }

        $boardId = $this->resolveBoardId($request);
        $scope = (string) ($request->query['scope'] ?? '');
        $targetId = (string) ($request->query['targetId'] ?? '');

        if ($boardId === null || !in_array($scope, ['item', 'list'], true) || $targetId === '') {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        $state = $this->catalog->getState($boardId, (int) $user['id']);
        if (!$state instanceof BoardState) {
            return Response::error('BOARD_NOT_FOUND', 'Board introuvable.', [], 404);
        }

        $tags = $this->tagSuggestions->allowedTags(
            $state,
            (string) $boardId,
            (string) $user['id'],
            $scope,
            $targetId
        );
        return Response::ok([
            'tags' => $tags,
            'rulesVersion' => $this->rulesVersion,
        ]);
    }

    public function export(Request $request): Response
    {
        $user = $request->getAttribute('user');
        if (!$user) {
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }

        $boardId = $this->resolveBoardId($request);
        if ($boardId === null) {
            return Response::error('INVALID_PAYLOAD', 'Données invalides.', [], 422);
        }

        $snapshot = $this->boards->getSnapshot((string) $boardId, (int) $user['id']);
        if ($snapshot === null) {
            return Response::error('BOARD_NOT_FOUND', 'Board introuvable.', [], 404);
        }

        $meta = $this->catalog->getMetadata($boardId, (int) $user['id']);
        if ($meta === null) {
            return Response::error('BOARD_NOT_FOUND', 'Board introuvable.', [], 404);
        }

        $rules = $this->boardRules->export($boardId, (int) $user['id']);

        return Response::ok([
            'board' => [
                'id' => (int) $meta['id'],
                'title' => (string) $meta['title'],
                'exportedAt' => gmdate('c'),
                'updatedAt' => isset($meta['updated_at']) ? (int) $meta['updated_at'] : null,
                'createdAt' => isset($meta['created_at']) ? (int) $meta['created_at'] : null,
                'revision' => $snapshot->revision(),
                'state' => $snapshot->state()->toArray(),
                'history' => $snapshot->history(),
                'rules' => $rules,
            ],
        ]);
    }

    // Removed: import migrated to Non-Board Bus (Board.Import)

    // Removed: autosave migrated to Non-Board Bus (Board.Autosave)

    private function resolveBoardId(Request $request): ?int
    {
        $attribute = $request->getAttribute('boardId');
        if ($attribute !== null) {
            $value = (int) $attribute;
            if ($value > 0) {
                return $value;
            }
        }

        $queryValue = (int) ($request->query['boardId'] ?? 0);
        if ($queryValue > 0) {
            return $queryValue;
        }

        return null;
    }
}
