<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Controllers;

use Skyboard\Application\Commands\CommandBus;
use Skyboard\Application\NonBoard\NonBoardBus;
use Skyboard\Application\Commands\CreateColumnCommand;
// legacy V2 list commands removed
use Skyboard\Application\Commands\CreateWorkspaceCommand;
use Skyboard\Application\Commands\CreateNodeCommand;
use Skyboard\Application\Commands\UpdateNodeCommand;
use Skyboard\Application\Commands\MoveNodeCommand;
use Skyboard\Application\Commands\DeleteNodeCommand;
use Skyboard\Application\Commands\AddTagV3Command;
use Skyboard\Application\Commands\RemoveTagV3Command;
use Skyboard\Application\Commands\MoveColumnCommand;
// legacy V2 list commands removed
use Skyboard\Application\Commands\RenameColumnCommand;
// legacy V2 list commands removed
use Skyboard\Application\Commands\RenameWorkspaceCommand;
// legacy V2 list commands removed
use Skyboard\Application\Commands\UpdateTagFilterCommand;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;
use Skyboard\Infrastructure\Logging\CommandAuditLogger;
use InvalidArgumentException;

final class CommandController
{
    public function __construct(
        private readonly CommandBus $bus,
        private readonly ?CommandAuditLogger $logger = null,
        private readonly ?NonBoardBus $nonBoard = null,
    )
    {
    }

    public function execute(Request $request): Response
    {
        $user = $request->getAttribute('user');
        $type = (string) ($request->body['type'] ?? '');
        $payloadRaw = $request->body['payload'] ?? [];
        // Support multipart (FormData) where payload is a JSON string
        if (is_string($payloadRaw)) {
            $decoded = json_decode($payloadRaw, true);
            $payload = is_array($decoded) ? $decoded : [];
        } elseif (is_array($payloadRaw)) {
            $payload = $payloadRaw;
        } else {
            $payload = [];
        }
        $boardId = (string) ($request->body['boardId'] ?? '');
        $expectedRevision = $request->body['revision'] ?? null;
        $expectedRevision = is_numeric($expectedRevision) ? (int) $expectedRevision : null;

        $corrHeader = $request->headers['X-Action-Correlation-Id'] ?? $request->headers['x-action-correlation-id'] ?? null;
        $corrId = is_string($corrHeader) && $corrHeader !== '' ? $corrHeader : null;
        $baseAudit = [
            'type' => $type,
            'boardId' => $boardId !== '' ? $boardId : null,
            'actorId' => isset($user['id']) ? (string) $user['id'] : null,
            'revision' => $expectedRevision,
            'ip' => $request->server['REMOTE_ADDR'] ?? null,
            'userAgent' => $request->headers['User-Agent'] ?? $request->headers['user-agent'] ?? null,
            'path' => $request->path,
            'correlationId' => $corrId,
        ];

        if (!$user) {
            $this->audit($baseAudit + ['status' => 'unauthorized', 'reason' => 'UNAUTHORIZED']);
            return Response::error('UNAUTHORIZED', 'Authentification requise.', [], 401);
        }
        if ($type === '') {
            $this->audit($baseAudit + ['status' => 'invalid_payload', 'reason' => 'INVALID_PAYLOAD']);
            return Response::error('INVALID_PAYLOAD', 'Commande invalide.', [], 422);
        }

        // Non-board commands (Account./File./PackDataset./Board.* gestion + Notifications/Activity)
        if ($this->nonBoard && (
            str_starts_with($type, 'Account.') ||
            str_starts_with($type, 'File.') ||
            str_starts_with($type, 'PackDataset.') ||
            str_starts_with($type, 'Board.') ||
            str_starts_with($type, 'Notification.') ||
            str_starts_with($type, 'NotificationRich.') ||
            str_starts_with($type, 'Activity.') ||
            str_starts_with($type, 'User.')
        )) {
            return $this->nonBoard->dispatch($type, $payload, $request, (int) $user['id']);
        }

        if ($boardId === '') {
            $this->audit($baseAudit + ['status' => 'invalid_payload', 'reason' => 'INVALID_PAYLOAD']);
            return Response::error('INVALID_PAYLOAD', 'Commande invalide (boardId requis).', [], 422);
        }
        $command = $this->mapCommand($type, (int) $user['id'], $boardId, $payload);
        if ($command === null) {
            $this->audit($baseAudit + ['status' => 'invalid_payload', 'reason' => 'UNKNOWN_COMMAND']);
            return Response::error('UNKNOWN_COMMAND', 'Commande inconnue.', [], 422);
        }
        try {
            $result = $this->bus->dispatch($command, $expectedRevision);
        } catch (InvalidArgumentException $invalid) {
            $this->audit($baseAudit + ['status' => 'invalid_payload', 'reason' => 'INVALID_ARGUMENT']);
            return Response::error('INVALID_PAYLOAD', $invalid->getMessage(), [], 422);
        }
        $autosave = $result['autosave'] ?? null;
        $durationMs = is_array($autosave) && isset($autosave['durationMs']) ? (int) $autosave['durationMs'] : null;
        $this->audit($baseAudit + [
            'status' => $result['status'] ?? 'unknown',
            'reason' => $result['error']['reasonId'] ?? null,
            'durationMs' => $durationMs,
        ]);
        $normalized = $this->normalizeResult($result);

        $status = (string) ($result['status'] ?? 'error');
        if ($status === 'ok') {
            return Response::ok($normalized);
        }

        // Map error-like statuses to HTTP status
        $http = match ($status) {
            'policy_veto', 'invariant_violation', 'conflict' => 409,
            'error' => 422,
            default => 500,
        };

        $code = (string) ($normalized['code'] ?? 'COMMAND_FAILED');
        $message = (string) ($normalized['message'] ?? 'Action refusée');
        $details = $normalized;
        unset($details['ok'], $details['code'], $details['message']);
        return Response::error($code, $message, $details, $http);
    }

    // Removed non-board glue (migrated to NonBoardBus)

    private function mapCommand(string $type, int $actorId, string $boardId, array $payload): ?object
    {
        return match ($type) {
            // v3 commands (no retro-compat fallbacks)
            'CreateNode' => isset($payload['parentId'])
                ? new CreateNodeCommand($boardId, (string) $actorId, (array) $payload)
                : null,
            'UpdateNode' => isset($payload['nodeId'])
                ? new UpdateNodeCommand($boardId, (string) $actorId, (string) $payload['nodeId'], (array) ($payload['changes'] ?? $payload))
                : null,
            'MoveNode' => isset($payload['nodeId'], $payload['toParentId'])
                ? new MoveNodeCommand($boardId, (string) $actorId, (string) $payload['nodeId'], (string) $payload['toParentId'], (int) ($payload['toIndex'] ?? 0))
                : null,
            'DeleteNode' => isset($payload['nodeId'])
                ? new DeleteNodeCommand($boardId, (string) $actorId, (string) $payload['nodeId'])
                : null,
            'AddTagV3' => isset($payload['nodeId'], $payload['tag'])
                ? new AddTagV3Command($boardId, (string) $actorId, (string) $payload['nodeId'], (array) $payload['tag'])
                : null,
            'RemoveTagV3' => isset($payload['nodeId'], $payload['key'])
                ? new RemoveTagV3Command($boardId, (string) $actorId, (string) $payload['nodeId'], (string) $payload['key'], isset($payload['value']) ? (string) $payload['value'] : null)
                : null,
            'CreateColumn' => isset($payload['workspaceId'])
                ? new CreateColumnCommand($boardId, (string) $actorId, (string) $payload['workspaceId'], (string) ($payload['title'] ?? 'Nouvelle colonne'))
                : null,
            'RenameColumn' => isset($payload['workspaceId'], $payload['columnId'], $payload['title'])
                ? new RenameColumnCommand($boardId, (string) $actorId, (string) $payload['workspaceId'], (string) $payload['columnId'], (string) $payload['title'])
                : null,
            'MoveColumn' => isset($payload['workspaceId'], $payload['columnId'])
                ? new MoveColumnCommand($boardId, (string) $actorId, (string) $payload['workspaceId'], (string) $payload['columnId'], (int) ($payload['position'] ?? 0))
                : null,
            'CreateWorkspace' => new CreateWorkspaceCommand($boardId, (string) $actorId, (string) ($payload['title'] ?? 'Workspace')),
            'RenameWorkspace' => isset($payload['workspaceId'], $payload['title'])
                ? new RenameWorkspaceCommand($boardId, (string) $actorId, (string) $payload['workspaceId'], (string) $payload['title'])
                : null,
            'UpdateTagFilter' => isset($payload['selected']) && is_array($payload['selected'])
                ? new UpdateTagFilterCommand($boardId, (string) $actorId, array_values($payload['selected']))
                : null,
            default => null,
        };
    }

    /**
     * @param array<string,mixed> $entry
     */
    private function audit(array $entry): void
    {
        $this->logger?->log($entry);
    }

    // Removed: jsonError in favor of Response::error

    /**
     * @param array<string,mixed> $result
     * @return array<string,mixed>
     */
    private function normalizeResult(array $result): array
    {
        $status = (string) ($result['status'] ?? 'unknown');
        $payload = ['ok' => $status === 'ok', 'status' => $status];
        foreach (['trace', 'autosave', 'events', 'finalPatch', 'patch', 'revision', 'lastSavedAt', 'rulesVersion'] as $key) {
            if (array_key_exists($key, $result)) {
                $payload[$key] = $result[$key];
            }
        }

        if ($payload['ok']) {
            return $payload;
        }

        $error = is_array($result['error'] ?? null) ? $result['error'] : [];
        $code = (string) ($error['code'] ?? 'COMMAND_FAILED');
        $message = (string) ($error['message'] ?? 'Action refusée');
        $normalizedError = ['code' => $code, 'message' => $message];
        if (isset($error['reasonId'])) {
            $normalizedError['reasonId'] = $error['reasonId'];
        }
        if (isset($error['details']) && is_array($error['details'])) {
            $normalizedError['details'] = $error['details'];
        }
        $payload['error'] = $normalizedError;
        $payload['code'] = $code;
        $payload['message'] = $message;

        return $payload;
    }
}
