<?php

declare(strict_types=1);

namespace Skyboard\Interfaces\Http\Middleware;

use Closure;
use Skyboard\Infrastructure\Http\Request;
use Skyboard\Infrastructure\Http\Response;
use Throwable;

final class ErrorHandlingMiddleware
{
    public function __construct(
        private readonly string $environment = 'prod'
    ) {
    }

    public function __invoke(Request $request, Closure $next): Response
    {
        try {
            return $next($request);
        } catch (Throwable $throwable) {
            $status = $this->resolveStatus($throwable);
            $code = $this->resolveCode($throwable, $status);
            $message = $this->resolveMessage($throwable, $status);
            $details = $this->resolveDetails($throwable) ?? [];

            $this->log($throwable, $request);

            return Response::error($code, $message, $details, $status);
        }
    }

    private function resolveStatus(Throwable $throwable): int
    {
        $code = $throwable->getCode();
        if (is_int($code) && $code >= 400 && $code <= 599) {
            return $code;
        }

        return 500;
    }

    private function resolveCode(Throwable $throwable, int $status): string
    {
        if ($status >= 500) {
            return 'INTERNAL_SERVER_ERROR';
        }

        $code = $throwable->getCode();
        if (is_string($code) && $code !== '') {
            return $code;
        }

        return strtoupper(str_replace(' ', '_', $throwable->getMessage() ?: 'UNHANDLED_ERROR'));
    }

    private function resolveMessage(Throwable $throwable, int $status): string
    {
        if ($status >= 500 && $this->environment !== 'dev') {
            return 'Une erreur inattendue est survenue.';
        }

        $message = $throwable->getMessage();
        return $message !== '' ? $message : 'Erreur inattendue';
    }

    /**
     * @return array<string,mixed>|null
     */
    private function resolveDetails(Throwable $throwable): ?array
    {
        if ($this->environment !== 'dev') {
            return null;
        }

        return [
            'exception' => $throwable::class,
            'file' => $throwable->getFile(),
            'line' => $throwable->getLine(),
        ];
    }

    private function log(Throwable $throwable, Request $request): void
    {
        $message = sprintf('[HTTP] %s %s failed: %s', $request->method, $request->path, $throwable->getMessage());
        error_log($message . "\n" . $throwable);
    }
}
