#!/usr/bin/env bash
set -euo pipefail

# Simple post-deploy smoke for Skyboard V4
# Requirements: curl; optional: jq for JSON parsing

BASE_URL="${BASE_URL:-http://localhost}"
EMAIL="${EMAIL:-smoke.$(date +%s)@example.test}"
PASSWORD="${PASSWORD:-Test12345!}"
COOKIES="${COOKIES:-/tmp/sb_cookies_$$.txt}"

cleanup() { rm -f "$COOKIES" >/dev/null 2>&1 || true; }
trap cleanup EXIT

header() { echo; echo "[SMOKE] $*"; }

json() { jq -r "$1" 2>/dev/null || true; }

req_json() {
  local method="$1"; shift
  local url="$1"; shift
  local data="${1:-}"; shift || true
  curl -sS -b "$COOKIES" -c "$COOKIES" \
    -H 'Accept: application/json' \
    ${data:+-H 'Content-Type: application/json'} \
    -X "$method" \
    ${data:+--data "$data"} \
    "$url"
}

csrf_token=""
refresh_csrf() {
  local raw
  raw=$(req_json GET "$BASE_URL/api/session")
  if command -v jq >/dev/null 2>&1; then
    csrf_token=$(echo "$raw" | json '.csrf // .CSRF // empty')
  else
    csrf_token=$(echo "$raw" | sed -n 's/.*"csrf":"\([^"]*\)".*/\1/p' | head -n1)
  fi
  if [ -z "$csrf_token" ]; then
    echo "[WARN] CSRF token not found in /api/session response" >&2
  fi
}

post_command() {
  local type="$1"; shift
  local payload="$1"; shift
  curl -sS -b "$COOKIES" -c "$COOKIES" \
    -H 'Accept: application/json' \
    -H 'Content-Type: application/json' \
    -H "X-CSRF-Token: $csrf_token" \
    -X POST \
    --data "{\"type\":\"$type\",\"payload\":$payload}" \
    "$BASE_URL/api/commands"
}

header "Session"
req_json GET "$BASE_URL/api/session" | tee /dev/stderr >/dev/null

header "Login (or register)"
login_body="{\"email\":\"$EMAIL\",\"password\":\"$PASSWORD\"}"
login=$(req_json POST "$BASE_URL/api/auth/login" "$login_body" || true)
if echo "$login" | grep -qi 'ok":false'; then
  header "Register"
  req_json POST "$BASE_URL/api/auth/register" "$login_body" >/dev/null
fi
refresh_csrf

header "Boards: create"
created=$(post_command 'Board.Create' '{"title":"Smoke Board"}')
echo "$created" | tee /dev/stderr >/dev/null
board_id=""
if command -v jq >/dev/null 2>&1; then
  board_id=$(echo "$created" | json '.board.id // empty')
else
  board_id=$(echo "$created" | sed -n 's/.*"id":\([0-9][0-9]*\).*/\1/p' | head -n1)
fi
if [ -z "$board_id" ]; then echo "[ERROR] board id not found" >&2; exit 1; fi

header "Boards: state"
state=$(req_json GET "$BASE_URL/api/boards/$board_id/state")
echo "$state" | tee /dev/stderr >/dev/null
revision=""
if command -v jq >/dev/null 2>&1; then
  revision=$(echo "$state" | json '.meta.revision // empty')
else
  revision=$(echo "$state" | sed -n 's/.*"revision":\([0-9][0-9]*\).*/\1/p' | head -n1)
fi

header "Boards: autosave"
autosave=$(post_command 'Board.Autosave' "{\"boardId\":\"$board_id\",\"revision\":$revision}")
echo "$autosave" | tee /dev/stderr >/dev/null

header "Files: compose markdown"
compose=$(post_command 'File.Compose' '{"name":"note.md","kind":"text","format":"markdown","content":"# Titre"}')
echo "$compose" | tee /dev/stderr >/dev/null
file_id=""
if command -v jq >/dev/null 2>&1; then
  file_id=$(echo "$compose" | json '.file.id // .file.publicId // empty')
else
  file_id=$(echo "$compose" | sed -n 's/.*"id":"\([^"]*\)".*/\1/p' | head -n1)
fi

header "Files: update content"
update=$(post_command 'File.UpdateContent' "{\"fileId\":\"$file_id\",\"kind\":\"text\",\"format\":\"markdown\",\"content\":\"Smoke update\"}")
echo "$update" | tee /dev/stderr >/dev/null

header "Files: read JSON content"
read=$(curl -sS -b "$COOKIES" -c "$COOKIES" -H 'Accept: application/json' "$BASE_URL/api/files/$file_id/content")
echo "$read" | tee /dev/stderr >/dev/null

header "Packs: datasets (categories)"
ds=$(req_json GET "$BASE_URL/api/packs/datasets/org%3Acategories")
echo "$ds" | tee /dev/stderr >/dev/null

echo
echo "[SMOKE] Done. If outputs show ok:true and expected fields, smoke passed."

